// ConfigWriter.cpp: implementation of the CConfigWriter class.
//
//////////////////////////////////////////////////////////////////////

#include "pch.h"
#include "Common.hpp"
#include "Temp.hpp"
#include "ConfigWriter.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


CConfigWriter::CConfigWriter(const tstring& aFileName) :
	fileName(aFileName), closed(false), resolveEntities(false), resolvedData(),
	oemData(false)
{
	// extract path
	tstring	path, prefix;

	int	lastSeparatorPos = fileName.find_last_of( PATH_SEPARATOR );
	if (lastSeparatorPos >= 0) {
		path   = fileName.substr( 0, lastSeparatorPos );
		prefix = fileName.substr( lastSeparatorPos + 1 );
	}
	else {
		// path is empty
		prefix = fileName;
	}

	tempFileName = getTempFileName( path, prefix );
	if (tempFileName.length() <= 0) {
		// TODO: raise exception
	}

	out.open( tempFileName.c_str() );
	// TODO: check result of file opening
}


CConfigWriter::~CConfigWriter()
{
	if (!closed)
		close();
}


void CConfigWriter::close() {
	if (!closed) {
		out.close();

		g_vars.m_log << CLog::ResetSingle( new CLog::StdOut() );

		if (_tunlink( fileName.c_str() ) == 0 || errno == ENOENT) {
			if (_trename( tempFileName.c_str(), fileName.c_str() ) == 0) {
				if (!::SetFileAttributes( fileName.c_str(), FILE_ATTRIBUTE_NORMAL )) {
					// we failed to set normal attribute
					g_vars.m_log << TEXT("Failed to set normal attribute for the configuration file:\n")
								 << TEXT("  file name = ") << TEXT("\"") << fileName << TEXT("\"\n");
				}
			} else {
				// we failed to rename temp name to the real name
				g_vars.m_log << TEXT("Failed to rename temporary file into real configuration file:\n")
							 << TEXT("  temp file name = ") << TEXT("\"") << tempFileName << TEXT("\"\n")
							 << TEXT("  real file name = ") << TEXT("\"") << fileName << TEXT("\"\n");
			}
		} else {
			// we failed to remove old configuration file
			g_vars.m_log << TEXT("Failed to remove old configuration file:\n")
						 << TEXT("  file name = ") << TEXT("\"") << fileName << TEXT("\"\n");
		}

		g_vars.m_log << CLog::Pop(3);

		closed = true;

	} else {
		// TODO: raise exception
	}
}


void CConfigWriter::cancel() {
	if (!closed) {
		out.close();
		remove( tempFileName.c_str() );
	}
}


CConfigWriter& CConfigWriter::operator <<(const tstring& data) {
	print( data.c_str() );

	return (*this);
}


CConfigWriter& CConfigWriter::operator <<(LPCTSTR data) {
	print( data );

	return (*this);
}


CConfigWriter& CConfigWriter::operator <<(const int data) {
	TCHAR	buffer[128];
	_stprintf( buffer, TEXT("%d"), data );
	print( buffer );

	return (*this);
}


CConfigWriter& CConfigWriter::operator <<(const CEntityResolver& resolver) {
	resolveEntities = true;
	return (*this);
}


CConfigWriter& CConfigWriter::operator <<(const COemManipulator& oem) {
	oemData = true;
	return (*this);
}


void CConfigWriter::print(LPCTSTR data) {
	if (resolveEntities) {
		resolve( data );
		data = resolvedData.c_str();
		resolveEntities = false;
	}

	if (oemData) {
		LPSTR	oemBuffer = new char[lstrlen(data)+1];
		::CharToOem( data, oemBuffer );
		out << oemBuffer;
		delete [] oemBuffer;
		oemData = false;
	}
	else {
		out << data;
	}
}


void CConfigWriter::resolve(LPCTSTR data) {
	// reset resolved data
	resolvedData = TEXT("");

	tstring	entity;
	int		startPos = 0;
	int		endPos   = 0;
	while (data[endPos] != TEXT('\0')) {
		bool	resolveTime = true;	// nice name? ;)
		switch (data[endPos]) {
		case '<':
			entity = TEXT("&lt;");
			break;
		case '>':
			entity = TEXT("&gt;");
			break;
		case '"':
			entity = TEXT("&quot;");
			break;
		case '\'':
			entity = TEXT("&apos;");
			break;
		case '&':
			entity = TEXT("&amp;");
			break;
		default:
			resolveTime = false;
		}
		if (resolveTime) {
			if (startPos < endPos) {
				resolvedData.append( data + startPos, endPos - startPos );
			}
			resolvedData += entity;
			++endPos;
			startPos = endPos;
		}
		else {
			++endPos;
		}
	}
	if (startPos < endPos) {
		resolvedData.append( data + startPos, endPos - startPos );
	}
}
